/*!
 * @nuxt/cli v2.18.1 (c) 2016-2024
 * Released under the MIT License
 * Repository: https://github.com/nuxt/nuxt.js
 * Website: https://nuxtjs.org
*/
'use strict';

const consola = require('consola');
const utils = require('@nuxt/utils');
const index = require('./cli-index.js');
require('@nuxt/config');
require('path');
require('exit');
require('chalk');
require('std-env');
require('wrap-ansi');
require('boxen');
require('minimist');
require('hookable');
require('defu');
require('semver');
require('fs');
require('execa');

const build = {
  name: "build",
  description: "Compiles the application for production deployment",
  usage: "build <dir>",
  options: {
    ...index.common,
    ...index.locking,
    analyze: {
      alias: "a",
      type: "boolean",
      description: "Launch webpack-bundle-analyzer to optimize your bundles",
      prepare(cmd, options, argv) {
        options.build = options.build || {};
        if (argv.analyze && typeof options.build.analyze !== "object") {
          options.build.analyze = true;
        }
      }
    },
    devtools: {
      type: "boolean",
      default: false,
      description: "Enable Vue devtools",
      prepare(cmd, options, argv) {
        options.vue = options.vue || {};
        options.vue.config = options.vue.config || {};
        if (argv.devtools) {
          options.vue.config.devtools = true;
        }
      }
    },
    generate: {
      type: "boolean",
      default: true,
      description: "Don't generate static version for SPA mode (useful for nuxt start)"
    },
    quiet: {
      alias: "q",
      type: "boolean",
      description: "Disable output except for errors",
      prepare(cmd, options, argv) {
        options.build = options.build || {};
        if (argv.quiet) {
          options.build.quiet = Boolean(argv.quiet);
        }
      }
    },
    standalone: {
      type: "boolean",
      default: false,
      description: "Bundle all server dependencies (useful for nuxt-start)",
      prepare(cmd, options, argv) {
        if (argv.standalone) {
          options.build.standalone = true;
        }
      }
    }
  },
  async run(cmd) {
    const config = await cmd.getNuxtConfig({ dev: false, server: false, _build: true });
    config.server = (config.mode === utils.MODES.spa || config.ssr === false) && cmd.argv.generate !== false;
    const nuxt = await cmd.getNuxt(config);
    if (cmd.argv.lock) {
      await cmd.setLock(await index.createLock({
        id: "build",
        dir: nuxt.options.buildDir,
        root: config.rootDir
      }));
    }
    if (nuxt.options.mode === utils.MODES.spa && nuxt.options.target === utils.TARGETS.server && cmd.argv.generate !== false) {
      const generator = await cmd.getGenerator(nuxt);
      await generator.generate({ build: true });
    } else {
      const builder = await cmd.getBuilder(nuxt);
      await builder.build();
      const nextCommand = nuxt.options.target === utils.TARGETS.static ? "nuxt generate" : "nuxt start";
      consola.info("Ready to run `" + nextCommand + "`");
    }
  }
};

exports.default = build;
