<?php

namespace App\Traits\Logs;


use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Spatie\Activitylog\Models\Activity as ModelsActivity;


trait HasCustomLogs
{

    /**
     *
     * This function logs a custom message with relevant information such as log name, performed on
     * model, message, properties, folder log type, and activity. Example usage:
     * * $this->logCustomMessage('department_created', $department, 'Department has been created', $department, 'Create a department', new Activity());
     * @param String logname A string representing the name of the log to which the message will be
     * logged. This should be a unique identifier for the log and follows a snake_case convention.
     * @param Model performedOn This parameter is an optional Model object that represents the entity
     * on which the action is being performed. For example, if a user is updating a blog post, the blog
     * post would be the "performedOn" entity. If this parameter is not provided, it will be set to
     * null.
     * @param String message The message that will be logged. It is a required parameter.
     * @param object properties An optional object that contains additional information to be logged
     * along with the message. This can be used to provide context or additional details about the
     * event being logged.
     * @param string logType This parameter is a string that represents the type of log that is
     * being created and short description of the logname. If it is not provided, the default value of 'default' will be used.
     * @param ModelsActivity activity The  parameter is an instance of the ModelsActivity
     * model, which is used to create a new activity log entry in the database.
     * @return the result of creating a new record in the `ModelsActivity` model with the provided
     * parameters.
     */
    public function logCustomMessage(String $logname, Model $performedOn = null, String $message, object $properties = null, string $logType, ModelsActivity $activity)
    {
        return $activity::create([
            'log_name' => $logname,
            'type' => $logType ?? 'default',
            'description' => $message ?? '',
            'subject_type' => $performedOn ? get_class($performedOn) : null,
            'subject_id' => $performedOn ? $performedOn->id : null,
            'causer_type' => get_class(Auth::user()),
            'causer_id' => Auth::user()->id,
            'properties' => $properties,
        ]);
    }
}
