<?php

namespace App\Models\AccessManagement\UserManagement;

use App\Models\User;
use App\Traits\Encryptable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class License extends Model
{
    use HasFactory, Encryptable;

    protected $fillable = [
        'user_id',
        'license_number',
        'e_signature',
        'blur_e_signature',
        'date_of_registration',
        'is_registered_license'
    ];

    protected $encryptable = [
        'license_number',
        'e_signature',
        'blur_e_signature'
    ];

    protected $with = [
        'user'
    ];

    public function __construct(array $attributes = []){
        $this->table = env('DB_DATABASE') . '.licenses';
        parent::__construct($attributes);
    }

    public function user(){
        return $this->belongsTo(User::class, 'user_id', 'id');
    }

    /**
     * It takes a number or an IP address and converts it to a license key
     * 
     * @param suffix This is the suffix that will be appended to the license key. It can be a user ID,
     * an IP address, or any other string.
     * 
     * @return A license string.
     */
    public static function generateLicense($suffix = null) {
        // Default tokens contain no "ambiguous" characters: 1,i,0,o
        if(isset($suffix)){
            // Fewer segments if appending suffix
            $num_segments = 3;
            $segment_chars = 6;
        }else{
            $num_segments = 4;
            $segment_chars = 5;
        }
        $tokens = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $license_string = '';
        // Build Default License String
        for ($i = 0; $i < $num_segments; $i++) {
            $segment = '';
            for ($j = 0; $j < $segment_chars; $j++) {
                $segment .= $tokens[rand(0, strlen($tokens)-1)];
            }
            $license_string .= $segment;
            if ($i < ($num_segments - 1)) {
                $license_string .= '-';
            }
        }
        // If provided, convert Suffix
        if(isset($suffix)){
            if(is_numeric($suffix)) {   // Userid provided
                $license_string .= '-'.strtoupper(base_convert($suffix,10,36));
            }else{
                $long = sprintf("%u\n", ip2long($suffix),true);
                if($suffix === long2ip($long) ) {
                    $license_string .= '-'.strtoupper(base_convert($long,10,36));
                }else{
                    $license_string .= '-'.strtoupper(str_ireplace(' ','-',$suffix));
                }
            }
        }
        return $license_string;
    }

    public static function generatePassword(){
        return bin2hex(openssl_random_pseudo_bytes(4));
    }
}
