<?php

namespace App\Http\Controllers;

use App\Enums\ExitInterviewStatus;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

/**
 * Controller used for external APIs 
 * Used by SRIT to get data from Employee Profile 
 */

class EmployeeAPIController extends Controller
{
    private $prefix = [
        'Ms',  // Female, not married
        'Mr',  // Male
        'Mrs', // Female, married
        'Dr',  // On graduate studies - degree - column basic_education
        'Miss', // Female, Not sure if married or not
    ];

    /**
     * Get Employee Profile Data
     */
    public function getEmployeeData(Request $request) {

        $validate = Validator::make($request->all(), [
            'employeeNo'  => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $user = User::whereHas('employeeMetaInfo', function($query) use ($request) {
                $query->where('employee_id', $request->employeeNo);
            })
            ->with([
                'personalInformation.employeeExtraFieldColumn' => function($query) {
                    $query->whereIn('field_name', [
                        'first_name',
                        'last_name',
                        'middle_name',
                        'date_of_birth',
                    ]);
                },
                'pdsChoices.employeeExtraFieldColumn' => function($query) {
                    $query->whereIn('field_name', [
                        'sex',
                        'civil_status'
                    ]);
                },
                'spouse',
                'exitInterview' => function($query) {
                    $query->where('status', ExitInterviewStatus::DONE);
                }
            ])
            ->without(['roles', 'permissions'])
            ->first();

            // Personal Information 

            $column = $user->personalInformation->employeeExtraFieldColumn->toArray();
            $pds_choices = $user->pdsChoices->employeeExtraFieldColumn->toArray();

            $column = json_decode(json_encode($column));
            $pds_choices = json_decode(json_encode($pds_choices));
            
            $first_name = array_search('first_name', array_column($column, 'field_name'));
            $last_name = array_search('last_name', array_column($column, 'field_name'));
            $middle_name = array_search('middle_name', array_column($column, 'field_name'));
            $dob = array_search('date_of_birth', array_column($column, 'field_name'));
            $sex = array_search('sex', array_column($pds_choices, 'field_name'));
            $civil_status = array_search('civil_status', array_column($pds_choices, 'field_name'));

            // Resigned Status
            $status = $user->exitInterview ? 'Inactive' : 'Active';

            // Prefix conditions
            $prefixind = $pds_choices[$sex]->field_value == 'Male' ? 1 : 4;
            $doctor = $user->graduate()
                ->whereHas('employeeExtraFieldColumn', function($query) {
                    $query->where('field_name', 'basic_education')
                        ->where('field_value', 'Doctoral');
                })->first() ? true : false;
            $married = $pds_choices[$civil_status]->field_value == 'Married';

            if($doctor) {
                $prefixind = 3; // Dr
            } else if($prefixind === 4 && ($civil_status !== false && !in_array($pds_choices[$civil_status]->field_value, ['Single', 'Married'])) ) {
                $prefixind = 0; // Ms
            } else if($prefixind === 4 && $married) {
                $prefixind = 2; // Mrs - if female and not doctor
            }

            $data = [
                'employee_no' => $user->employeeMetaInfo->employee_id, 
                'department' => $user->employeeMetaInfo->department ? $user->employeeMetaInfo->department->name : 'N/A', 
                'position' => $user->employeeMetaInfo->position ? $user->employeeMetaInfo->position->title : 'N/A', 
                'first_name' => $first_name !== false ? $column[$first_name]->field_value : 'N/A', 
                'last_name' => $last_name !== false ? $column[$last_name]->field_value : 'N/A', 
                'middle_name' => $middle_name !== false ? $column[$middle_name]->field_value : 'N/A', 
                'dob' => $dob !== false ? $column[$dob]->field_value : 'N/A', 
                'gender' => $sex !== false ? $pds_choices[$sex]->field_value : 'N/A', 
                'prefix' => $this->prefix[$prefixind],
                'status' => $status
            ];

            return response()->json([
                'data'  => $data
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

}
