<?php

namespace App\Http\Controllers\DeveloperTools;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Http\Controllers\Controller;
use App\Models\Activity;
use Illuminate\Http\Request;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use Carbon\Carbon;
use Exception;
use Validator;
use Auth;
use DB;

class UsersController extends Controller
{
    use HasCustomLogs;
    public function init_list(){
        return response()->json([
            'data'  =>  User::orderBy('id', 'desc')->paginate(10)
        ]);
    }

    public function search_list(Request $req){
        return response()->json([
            'data'  =>  User::where('name', 'LIKE', '%'.$req->keyword.'%')->orWhere('email', 'LIKE', '%'.$req->keyword.'%')->orderBy('id', 'desc')->paginate(10)
        ]);
    }

    public function create(Request $req){
        $valid = Validator::make($req->all(),[
            'name'  =>  'required|string|min:4',
            'email' =>  'required|email|unique:users',
            'password' => 'required|string|min:8',
            'roles' =>  'required|array',
            'permissions'   =>  'nullable|array',
            'verified'  =>  'required|boolean'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            $verified = $req->verified ? Carbon::now() : null;
            $u = User::create([
                'name'  =>  $req->name,
                'email' =>  $req->email,
                'password'  =>  bcrypt($req->password),
                'email_verified_at' =>  $verified
            ]);

            $this->logCustomMessage('user_created', $u, Auth::user()->name . ' created a new user ' . $u->name, $u, 'Create a user', new Activity());

            //send Notification
            if($verified === null){
                $u->sendEmailVerificationNotification();
            }
            //assign role
            $roles = array_column($req->roles, 'name');
            $u->assignRole($roles);
            //assign permissions
            if(count($req->permissions)){
                $perms = array_column($req->permissions, 'name');
                $u->syncPermissions($perms);
            }
            DB::commit();
            return response()->json([
                'text'  =>  'User has been created.'
            ]);
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in creating new user.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function update(Request $req){
        $valid = Validator::make($req->all(),[
            'name'  =>  'required|string|min:4',
            'email' =>  'required|email|unique:users,email,'.$req->id,
            'password' => 'nullable|string|min:8',
            'roles' =>  'required|array',
            'permissions'   =>  'nullable|array',
            'verified'  =>  'required|boolean'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $verified = $req->verified ? Carbon::now() : null;
                $user = User::findOrFail($req->id);

                $oldUser = clone $user;

                $user->name = $req->name;
                $user->email = $req->email;
                $user->email_verified_at = $verified;
                //check if they want to change pass.
                if($req->password){
                    $user->password = bcrypt($req->password);
                }
                //assign new role
                $roles =array_column($req->roles,'name');
                $user->syncRoles($roles);
                //check if permission to be updated
                if(count($req->permissions)){
                    $perms = array_column($req->permissions, 'name');
                    $user->syncPermissions($perms);
                }
                $user->save();
                //send Notification
                if($verified === null){
                    $user->sendEmailVerificationNotification();
                }

                $user->attributes = collect($user);
                $user->old = collect($oldUser);

                $this->logCustomMessage('user_updated', $user, Auth::user()->name . ' updated a user ' . $user->name, $user, 'Update a user', new Activity());

                DB::commit();
                return response()->json([
                    'text'  =>  'User has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'User doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a user.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function ban_process(Request $req){
        DB::beginTransaction();
        try{
            try{
                $user = User::findOrFail($req->id);
                if($req->status === 'ban'){
                    $user->revokePermissionTo('can login');
                }else{
                    $user->givePermissionTo('can login');
                }

                $message = $req->status === 'ban' ? 'banned a user' : 'unbanned a user';
                $logName = $req->status === 'ban' ? 'user_banned' : 'user_unbanned';

                $this->logCustomMessage($logName, $user, Auth::user()->name . " {$message} " . $user->name, $user, 'Ban a user', new Activity());

                DB::commit();
                return response()->json([
                    'text'  =>  'User has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'User doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a user.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

}
