<?php

namespace App\Http\Controllers\DeveloperTools;

use DB;
use Auth;
use Exception;
use Validator;
use App\Models\User;
use App\Models\Activity;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class RolesController extends Controller
{

    use HasCustomLogs;

    public function init_list(){
        return response()->json([
            'data'  =>  Role::orderBy('id', 'desc')->paginate(10)
        ]);
    }

    public function search_list(Request $req){
        return response()->json([
            'data'  =>  Role::where('name', 'LIKE', '%'.$req->keyword.'%')->orderBy('id', 'desc')->paginate(10)
        ]);
    }

    public function create(Request $req){
        $valid = Validator::make($req->all(),[
            'name'  =>  'required|string|min:4|unique:roles'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            $u = Role::create([
                'name'  =>  $req->name,
                'guard_name' =>  'api'
            ]);

            $this->logCustomMessage('role_created', $u, Auth::user()->name . ' created a new role ' . $u->name, $u, 'Create a role', new Activity());

            DB::commit();
            return response()->json([
                'text'  =>  'Role has been created.'
            ]);
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in creating new Role.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function update(Request $req){
        $valid = Validator::make($req->all(),[
            'name'  =>  'required|string|min:4|unique:roles,name,'.$req->id,
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $role = Role::findOrFail($req->id);

                $oldRole = clone $role;

                $role->name = $req->name;
                $role->save();

                $role->attributes = collect($role);
                $role->old = collect($oldRole);

                $this->logCustomMessage('role_updated', $role, Auth::user()->name . " updated a role {$oldRole->name} to {$role->name}", $role, 'Update a role', new Activity());

                DB::commit();
                return response()->json([
                    'text'  =>  'Role has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Role doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Role.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete(Request $req){
        DB::beginTransaction();
        try{
            $perm = Role::findOrFail($req->id);

            $this->logCustomMessage('role_deleted', $perm, Auth::user()->name . " deleted a role {$perm->name}", $perm, 'Delete a role', new Activity());

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Role has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Role doesnt exists.' ],
            ],400);
        }
    }

}
