<?php

namespace App\Http\Controllers\Auth;

use Throwable;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use MikeMcLin\WpPassword\Facades\WpPassword;
use App\Services\AccountSyncing\WpAccountSyncService;
use App\Models\AccessManagement\UserManagement\License;

class UserController extends Controller
{
    /**
     * Get authenticated user.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function current(Request $request)
    {

        $license = License::where([
            ['user_id', Auth::id()],
            ['is_registered_license', false]
        ])->first();

        $isSupderAdmin = Auth::user()->hasRole(['Super Admin']);

        if ($license) {
            $request->user()['license_id'] = $license->id;
            $request->user()['is_ask_signature'] = true;
        }

        if ($isSupderAdmin) {
            $request->user()['is_super_admin'] = true;
        }
        return response()->json($request->user());
    }

    public function fetchUserWithLicense()
    {
        return response()->json([
            'data' => User::whereHas('license')->get()
        ]);
    }

    public function fetchUsers(Request $request)
    {
        // return User::whereNotIn('id', [1, 2])->when($request->with_info == 'false', function ($query) {
        //     $query->select('id', 'name', 'email')->without([
        //         'currentRole',
        //         'roles',
        //         'permissions',
        //         'storage',
        //         'supervisor',
        //         'user_supervisor',
        //         'exitInterview',
        //         'userProfilePicture',
        //         'profileBasicInfo'
        //     ])->with([
        //         'employeeMetaInfo' => function ($query) {
        //             $query->select('id', 'user_id', 'employee_id')->without([
        //                 'corporation',
        //                 'branch',
        //                 'division',
        //                 'department',
        //                 'employeeType',
        //                 'position',
        //                 'itemCode'
        //             ]);
        //         }
        //     ]);
        // })->when(isset($request->with_department) && $request->with_department == 'true', function ($query) use ($request) {
        //     $query->whereHas('employeeMetaInfo')
        //         ->with([
        //             'employeeMetaInfo' => function ($query) {
        //                 $query->select('id', 'user_id', 'employee_id', 'department_id')
        //                     ->without(
        //                         'corporation',
        //                         'branch',
        //                         'division',
        //                         'employeeType',
        //                         'position',
        //                         'itemCode'
        //                     );
        //             }
        //         ]);
        // })
        //     ->when(isset($request->status), function ($query) use ($request) {
        //         $query->where('status', $request->status);
        //     })->has('employeeMetaInfo')
        //     ->get();
        return User::whereNotIn('id', [1, 2])->when($request->with_info == 'false', function ($query) {
            $query->select('id', 'name', 'email')->without([
                'currentRole',
                'roles',
                'permissions',
                'storage',
                'supervisor',
                'user_supervisor',
                'exitInterview',
                'userProfilePicture',
                'profileBasicInfo'
            ]);
        })->get();
    }

    public function syncAccount(Request $request)
    {
        try {
            $user = User::with(['personalInformation'])->find(Auth::id());

            $first_name = $user->personalInformation()->first()->employeeExtraFieldColumn()->where('field_name', 'first_name')->first()->field_value;
            $last_name = $user->personalInformation()->first()->employeeExtraFieldColumn()->where('field_name', 'last_name')->first()->field_value;

            if (!Hash::check($request->password, $user->password)) {
                return response()->json([
                    'error' => 'Invalid password',
                ], 403);
            }

            $lowerFirstName = strtolower($first_name);

            WpAccountSyncService::syncAccountsToWordpress([
                'ID' => $user->id,
                'user_login' => preg_replace('/\s+/', '.', $lowerFirstName),
                'user_pass' => WpPassword::make($request->password),
                'user_nicename' => preg_replace('/\s+/', '.', $lowerFirstName),
                'user_email' => $user->email,
                'user_url' => '',
                'user_registered' => Carbon::now(),
                'user_activation_key' => '',
                'user_status' => 0,
                'display_name' => $first_name . ' ' . $last_name,
            ]);

            return response()->json([
                'message' => 'You account has been sync',
            ], 200);
        } catch (Throwable $e) {
            return $e;
            return response()->json([
                'error'    => __('responses.exception'),
            ], 500);
        }
    }

    public function fetchEmployees(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return User::role(['Employee'])->where('name', 'LIKE', '%' . $request->employeeName . '%')
            ->whereHas('employeeMetaInfo', function ($query) use ($request) {
                $query->when($request->employee_id != null && isset($request->employee_id) && $request->employee_id != '', function ($query) use ($request) {
                    if (count($request->employee_id) > 0) {
                        $query->whereIn('employee_id', $request->employee_id);
                    } else {
                        $query->where('employee_id', $request->employee_id);
                    }
                });
                $query->when($request->position != null && isset($request->position) && $request->position != '', function ($query) use ($request) {
                    $query->where('position_id', $request->position);
                });
                $query->when($request->department != null && isset($request->department) && $request->department != '', function ($query) use ($request) {
                    $query->where('department_id', $request->department);
                });
                $query->when($request->division != null && isset($request->division) && $request->division != '', function ($query) use ($request) {
                    $query->where('division_id', $request->division);
                });
            })
            ->when($request->status != null && isset($request->status) && $request->status != '', function ($query) use ($request) {
                $query->where('status', $request->status);
            })
            ->without(['storage', 'roles', 'permissions', 'supervisor', 'exitInterview'])->paginate($paginate);
    }

    public function listEmployees(Request $request)
    {
        return User::whereNotIn('id', [1, 2])->where('name', 'LIKE', '%' . $request->employeeName . '%')
            ->whereHas('employeeMetaInfo', function ($query) use ($request) {
                $query->when($request->employee_id != null && isset($request->employee_id) && $request->employee_id != '', function ($query) use ($request) {
                    if (count($request->employee_id) > 0) {
                        $query->whereIn('employee_id', $request->employee_id);
                    } else {
                        $query->where('employee_id', $request->employee_id);
                    }
                });
                $query->when($request->position != null && isset($request->position) && $request->position != '', function ($query) use ($request) {
                    $query->where('position_id', $request->position);
                });
                $query->when($request->department != null && isset($request->department) && $request->department != '', function ($query) use ($request) {
                    $query->where('department_id', $request->department);
                });
                $query->when($request->division != null && isset($request->division) && $request->division != '', function ($query) use ($request) {
                    $query->where('division_id', $request->division);
                });
            })
            ->when($request->status != null && isset($request->status) && $request->status != '', function ($query) use ($request) {
                $query->where('status', $request->status);
            })
            ->without(['storage', 'roles', 'permissions', 'supervisor', 'exitInterview'])->get();
    }

    //Methods for SRIT API (CREATED TO AVOID BREAKING THE API FOR DOLPHINS)

    public function index(Request $request)
    {
        $withInfo = $request->with_info == "true" ? [
            'employeeMetaInfo.division' => function ($query) {
                $query->select('id', 'name', 'description')->without([
                    'sections'
                ]);
            }, 'employeeMetaInfo.department' => function ($query) {
                $query->select('id', 'name', 'description')->without([
                    'divisions'
                ]);
            }
        ] : [];

        return User::select('id', 'name', 'email', 'status')->whereNotIn('id', [1, 2])->where('name', 'LIKE', '%' . $request->name . '%')
            ->has('employeeMetaInfo')
            ->when($request->status != null && isset($request->status) && $request->status != '', function ($query) use ($request) {
                $query->where('status', $request->status);
            })
            ->without([
                'currentRole',
                'roles',
                'permissions',
                'storage',
                'supervisor',
                'user_supervisor',
                'exitInterview',
                'userProfilePicture',
                'profileBasicInfo',
                'employeeMetaInfo'
            ])->with($withInfo)->get()->append(['first_name', 'middle_name', 'last_name', 'gender']);
    }

    public function show(Request $request, $userId)
    {
        $withInfo = $request->with_info == "true" ? [
            'employeeMetaInfo.division' => function ($query) {
                $query->select('id', 'name', 'description')->without([
                    'sections'
                ]);
            }, 'employeeMetaInfo.department' => function ($query) {
                $query->select('id', 'name', 'description')->without([
                    'divisions'
                ]);
            }
        ] : [];

        $user = User::select('id', 'name', 'email', 'status')->whereNotIn('id', [1, 2])->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo',
            'employeeMetaInfo'
        ])->with($withInfo)->find($userId);

        if (!$user) {
            return response()->json([
                'message' => 'User not found',
            ], 404);
        }

        return $user->append(['first_name', 'middle_name', 'last_name', 'gender']);
    }

    public function paginate(Request $request)
    {

        $withInfo = $request->with_info == "true" ? [
            'employeeMetaInfo.division' => function ($query) {
                $query->select('id', 'name', 'description')->without([
                    'sections'
                ]);
            }, 'employeeMetaInfo.department' => function ($query) {
                $query->select('id', 'name', 'description')->without([
                    'divisions'
                ]);
            }
        ] : [];

        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return User::select('id', 'name', 'email', 'status')->whereNotIn('id', [1, 2])->where('name', 'LIKE', '%' . $request->name . '%')
            ->has('employeeMetaInfo')
            ->when($request->status != null && isset($request->status) && $request->status != '', function ($query) use ($request) {
                $query->where('status', $request->status);
            })
            ->without([
                'currentRole',
                'roles',
                'permissions',
                'storage',
                'supervisor',
                'user_supervisor',
                'exitInterview',
                'userProfilePicture',
                'profileBasicInfo',
                'employeeMetaInfo'
            ])->with($withInfo)->paginate($paginate)->append(['first_name', 'middle_name', 'last_name', 'gender']);
    }
    public function fetchEmployeeWithServiceRecords(Request $request)
    {
        $user = User::select('id', 'name', 'email', 'status')
            ->whereHas('employeeMetaInfo', function ($query) use ($request) {
                $query->whereIn('employee_id', $request->employee_ids);
            })
            ->where('status', $request->status)
            ->with([
                'employeeMetaInfo' => function ($query) {
                    $query->select('id', 'user_id', 'employee_id', 'division_id', 'department_id', 'position_id', 'item_code_id', 'date_hired')
                        ->without([
                            'corporation',
                            'branch',
                            'employeeType',
                        ]);
                },
                'employeeMetaInfo.division' => function ($query) {
                    $query->select('id', 'name', 'description')->without([
                        'sections'
                    ]);
                },
                'employeeMetaInfo.department' => function ($query) {
                    $query->select('id', 'name', 'description')->without([
                        'divisions'
                    ]);
                },
                'employeeMetaInfo.position' => function ($query) {
                    $query->select('id', 'title');
                },
                'employeeMetaInfo.itemCode' => function ($query) {
                    $query->select('id', 'item_code', 'description')->without([
                        'position'
                    ]);
                },
                'serviceRecord' => function ($query) {
                    $query->select('id', 'user_id', 'start_date', 'end_date', 'position_id', 'salary', 'allowance', 'employee_type_id', 'division_id', 'remark_id');
                },
                'serviceRecord.position' => function ($query) {
                    $query->select('id', 'title');
                },
                'serviceRecord.employeeType' => function ($query) {
                    $query->select('id', 'title');
                },
                'serviceRecord.division' => function ($query) {
                    $query->select('id', 'name')->without([
                        'sections'
                    ]);
                },
                'serviceRecord.remark' => function ($query) {
                    $query->select('id', 'abbreviation', 'meaning');
                },
                'actualDesignation' => function ($query) {
                    $query->select('id', 'user_id', 'office_id',  'department_id', 'division_id', 'section_id', 'unit_id');
                },
                'actualDesignation.office' => function ($query) {
                    $query->select('id', 'name', 'description');
                },
                'actualDesignation.department' => function ($query) {
                    $query->select('id', 'name', 'description');
                },
                'actualDesignation.section' => function ($query) {
                    $query->select('id', 'name', 'description');
                },
                'actualDesignation.unit' => function ($query) {
                    $query->select('id', 'name', 'description');
                }
            ])->without([
                'currentRole',
                'roles',
                'permissions',
                'storage',
                'supervisor',
                'user_supervisor',
                'exitInterview',
                'userProfilePicture',
                'profileBasicInfo'
            ])->get();

        if ($user->isEmpty()) {
            return response()->json([
                'message' => 'User not found',
            ], 404);
        }

        return $user->append(['first_name', 'middle_name', 'last_name', 'gender']);
    }
}
