<?php

namespace App\Http\Controllers\Auth;

use App\Exceptions\VerifyEmailException;
use App\Http\Controllers\Controller;
use App\Models\AccessManagement\UserManagement\License;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Carbon\Carbon;
use App\Models\User;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    /**
     * Attempt to log the user into the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return bool
     */
    protected function attemptLogin(Request $request)
    {
        $loginInput = $request->email_or_employee_id;
        $password = $request->password;

        $user = User::whereHas('employeeMetaInfo', function($query) use ($loginInput) {
            $query->where('employee_id', $loginInput);
        })->first();

        if ($user && auth('api')->attempt(['email' => $user->email, 'password' => $password])) {
            $token = auth('api')->attempt(['email' => $user->email, 'password' => $password]);
        } else {
            // Login using employee ID
            $token = auth('api')->attempt(['email' => $loginInput, 'password' => $password]);
        }

        $AdCredentials = [
            'mail' => $request->email_or_employee_id,
            'password' => $request->password
        ];

        if (!$token) {
            $token = auth('open-ldap')->attempt($AdCredentials);
        }

        if (!$token) {
            $token = auth('active-directory')->attempt($AdCredentials);
        }

        if (!$token) {
            return false;
        }

        $user = $this->guard()->user();
        if ($user instanceof MustVerifyEmail && !$user->hasVerifiedEmail()) {
            return false;
        }

        $this->guard()->setToken($token);

        return true;
    }

    /**
     * Send the response after the user was authenticated.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    protected function sendLoginResponse(Request $request)
    {
        $this->clearLoginAttempts($request);

        $token = (string) $this->guard()->getToken();
        $expiration = $this->guard()->getPayload()->get('exp');
        $user = $this->guard()->user();
        if(!$user->can('can login')){
            return response()->json([
                'errors'    =>  [ 'You are currently banned for log in.' ]
            ],403);
        }
        //create login notification
        $notifable = [];
            //notify current user
            $notifable[] = $user;
        $this->createNotif('Login Notification', 'You just logged in today @ '.(date('F d, Y h:i:s a', strtotime(Carbon::now()))), 'information', $notifable);
        return response()->json([
            'role' => $user->roles,
            'token' => $token,
            'token_type' => 'bearer',
            'expires_in' => $expiration - time(),
        ]);
    }

    /**
     * Get the failed login response instance.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function sendFailedLoginResponse(Request $request)
    {
        $user = $this->guard()->user();
        if ($user instanceof MustVerifyEmail && ! $user->hasVerifiedEmail()) {
            throw VerifyEmailException::forUser($user);
        }

        throw ValidationException::withMessages([
            $this->username() => [trans('auth.failed')],
        ]);
    }

    /**
     * Log the user out of the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        $this->guard()->logout();
    }

    public function username()
    {
        return 'email_or_employee_id';
    }
}
