<?php

namespace App\Http\Controllers\AccessManagement\UserManagement;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\MailController;
use Illuminate\Support\Facades\Validator;
use SoareCostin\FileVault\Facades\FileVault;
use App\Models\AccessManagement\UserManagement\License;
use Spatie\Permission\Models\Permission;

class LicenseController extends Controller
{
    public static function create(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'number_of_license' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        $licenses = [];
        $numberOfLicense = $request->number_of_license;

        DB::beginTransaction();
        try {
            for ($i = 0; $i < $numberOfLicense; $i++) {
                $license = License::generateLicense();
                while (in_array($license, $licenses)) {
                    $license = License::generateLicense();
                }
                array_push($licenses, $license);
                License::create([
                    'license_number' => License::generateLicense(),
                    'date_of_registration' => Carbon::now()
                ]);
            }

            DB::commit();
            return response()->json([
                'text'  =>  'License has been created.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating new license.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete(Request $request)
    {
        try {
            License::whereIn('id', $request->deleted_licenses)->delete();
            return response()->json([
                'deleted' => $request->deleted_licenses,
            ],200);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['There is a problem deleting the licenses. Please try again later.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function downloadSignature(Request $request)
    {

        $license = License::where('user_id', $request->user_id)->first();
        if (!$license) {
            abort(404);
        }

        $path = $license->e_signature;

        $encryptedFilePath = $license->e_signature . '.enc';

        if ($request->is_blur_signature) {
            $path = $license->blur_e_signature;
            $encryptedFilePath = $license->blur_e_signature . '.enc';
        }

        if (!Storage::disk('e_signature')->has($encryptedFilePath)) {
            abort(404);
        }

        Storage::disk('tmp_encrypt_document')->put($encryptedFilePath, Storage::disk('e_signature')->get($encryptedFilePath));

        FileVault::disk('tmp_encrypt_document')->decrypt($encryptedFilePath);

        return response()->download(Storage::disk('tmp_encrypt_document')->path($path))->deleteFileAfterSend(true);
    }

    public function assignESignature(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'signature'  =>  'required|file|mimes:png|max:3000',
            'license_id' => 'required|exists:' . env('DB_CONNECTION') . '.licenses,id',
            'user_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $license = License::where([
                ['id', $request->license_id],
                ['user_id', $request->user_id]
            ])->first();

            if ($license) {
                $imgPath = Storage::disk('tmp_encrypt_document')->put($request->user_id, $request->signature);

                $path = $request->signature->move(public_path() . '/upload', $request->signature->getClientOriginalName());

                $blurredImg = Image::make($path);

                for ($i = 0; $i < 5; $i++) {
                    $blurredImg->pixelate();
                }
                $blurredImg->blur();

                $uuid = Str::uuid();

                $blurIMGPath = $request->user_id . '/' . $uuid . '.png';
                Storage::disk('tmp_encrypt_document')->put($blurIMGPath, $blurredImg->save($path));

                if ($imgPath && $blurIMGPath) {
                    FileVault::disk('tmp_encrypt_document')->encrypt($imgPath);
                    FileVault::disk('tmp_encrypt_document')->encrypt($blurIMGPath);
                }

                Storage::disk('e_signature')->put($request->user_id . '/' . basename(Storage::disk('tmp_encrypt_document')->path($imgPath . '.enc')), Storage::disk('tmp_encrypt_document')->get($imgPath . '.enc'));
                Storage::disk('e_signature')->put($blurIMGPath . '.enc', Storage::disk('tmp_encrypt_document')->get($blurIMGPath . '.enc'));

                File::delete($path);

                Storage::disk('tmp_encrypt_document')->delete($imgPath . '.enc');
                Storage::disk('tmp_encrypt_document')->delete($blurIMGPath . '.enc');

                $license->update([
                    'is_registered_license' => true,
                    'e_signature' => $imgPath,
                    'blur_e_signature' => $blurIMGPath
                ]);
                $license->save();
                DB::commit();
                return response()->json([
                    'text'  =>  'Signature has been created.'
                ]);
            } else {
                return response()->json([
                    'message'    =>  ['There is a problem in creating an e-signature.'],
                ], 403);
            }
        } catch (Exception $e) {
            DB::rollback();
            return $e;
            return response()->json([
                'errors'    =>  ['There is a problem in creating an e-signature..'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function resetESignature(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'user_id'  =>  'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'license_id' => 'required|exists:' . env('DB_CONNECTION') . '.licenses,id',
            'e_signature' => 'required|string',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $exists = Storage::disk('e_signature')->exists($request->e_signature);

            if ($exists) {
                Storage::disk('e_signature')->deleteDirectory($request->user_id);

                License::where([
                    ['id', $request->license_id],
                    ['user_id', $request->user_id]
                ])->update([
                    'e_signature' => null
                ]);

                DB::commit();
                return response()->json([
                    'text'  =>  'Signature has been reset.'
                ]);
            } else {
                return response()->json([
                    'text'  =>  'Signature does not exists.'
                ]);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in resetting a signature.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function assignUser(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'name'  =>  'required|string',
            'email' => 'required|email',
            'license_id' => 'required|exists:' . env('DB_CONNECTION') . '.licenses,id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $password = License::generatePassword();

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => bcrypt($password),
                'email_verified_at' => Carbon::now(),
            ]);

            $license = License::where('id', $request->license_id)->first();

            if ($license->user_id) {
                return response()->json([
                    'message'       =>  "This license is already been taken."
                ], 403);
            }

            License::where('id', $request->license_id)->update([
                'user_id' => $user->id,
                'date_of_registration' => Carbon::now(),
            ]);

            DB::commit();
            MailController::sendCredentials($request->name, $request->email, $password);
            $permissions = Permission::select("id")->get();
            $permissionIDs = [];
            foreach ($permissions as $value) {
                array_push($permissionIDs, $value['id']);
            }
            $user->syncPermissions($permissionIDs);
            $user->assignRole(['Admin']);
            return response()->json([
                'text'  =>  'License has been updated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating the license.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function renameLicense(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'user_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'email' => 'required|email',
            'old_name' => 'required|string',
            'new_name' => 'required|string',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            User::where([
                ['id', $request->user_id],
                ['email', $request->email],
                ['name', $request->old_name]
            ])->update([
                'name' => $request->new_name,
            ]);

            DB::commit();

            return response()->json([
                'text'  =>  'License name has been updated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating the license name.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function resendCredential(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'user_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'email' => 'required|email'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $password = License::generatePassword();

            User::where([
                ['id', $request->user_id],
                ['email', $request->email]
            ])->update([
                'password' => bcrypt($password),
            ]);

            DB::commit();
            MailController::sendCredentials($request->name, $request->email, $password);
            return response()->json([
                'text'  =>  'Credentials has been sent.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in resending credentials.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function paginate(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return License::where('is_registered_license', $request->is_registered_license)->paginate($paginate);
    }

    public function reset(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'user_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'license_id' => 'required|exists:' . env('DB_CONNECTION') . '.licenses,id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            License::where([
                ['id', $request->license_id],
                ['user_id', $request->user_id]
            ])->update([
                'user_id' => null,
                'e_signature' => null,
                'is_registered_license' => false,
            ]);

            User::find($request->user_id)->delete();

            DB::commit();

            return response()->json([
                'text'  =>  'License has been reset/deactivated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in resetting/deactivating the license.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }
}
