<?php

namespace App\Http\Controllers\AccessManagement\GroupManagement;

use Activity;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Models\AccessManagement\GroupManagement\Role;
use App\Models\AccessManagement\GroupManagement\System;
use App\Models\AccessManagement\GroupManagement\SubRole;
use App\Models\AccessManagement\GroupManagement\ModelHasRole;
use App\Models\AccessManagement\GroupManagement\Permission;
use App\Models\AccessManagement\GroupManagement\RoleHasPermission;
use App\Models\AccessManagement\GroupManagement\RoleHasSystem;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Enums\Log\GroupLogType;

class SubRoleController extends Controller
{

    use HasCustomLogs;

    public function create(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'system_id' => 'required|exists:' . env('DB_CONNECTION') . '.systems,id',
            'name' => 'required|unique:roles',
            'parent_role_id' => 'exists:' . env('DB_CONNECTION') . '.roles,id',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $role = Role::where('name', $req->name)->first();
            $system = System::where('id', $req->system_id)->first();

            if (!$role) {
                $role = Role::create([
                    'name'  =>  $req->name . ' ' . $system->name,
                    'guard_name' =>  'api'
                ]);
            }

            $members = [];
            foreach ($req->group_member as $member) {
                array_push($members, array(
                    'role_id' => $role->id,
                    'model_type' => 'App\Models\User',
                    'model_id' => $member,
                ));
            }
            ModelHasRole::insert($members);
            $subRole = SubRole::create([
                'parent_role_id' => $req->parent_role_id,
                'sub_role_id' => $role->id,
            ]);

            $withoutRoleOtherInfo = function ($query) {
                $query->without([
                    'roleLeader',
                    'sectionAccess',
                    'member'
                ])->with(['roleHasSystem' => function ($query) {
                    $query->without([
                        'role',
                    ]);
                }]);
            };

            $subRole = SubRole::with([
                'parentRole' => $withoutRoleOtherInfo,
                'subRole' => $withoutRoleOtherInfo
            ])->find($subRole->id);

            $this->logCustomMessage('create_subgroup', $subRole, Auth::user()->name . ' created a subgroup named ' . $subRole->subRole->name . ' in the group ' . $subRole->parentRole->name, $subRole, GroupLogType::CREATE_SUBGROUP, new Activity());

            DB::commit();
            return response()->json([
                'text' => 'Sub Group has been created.'
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors' => ['There is a problem in creating new Sub Group'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    public function delete(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }
        try {
            SubRole::where('id', $req->id)->delete();
            return response()->json([
                'message' => 'Sub Group is deleted'
            ], 200);
            DB::commit();
            return response()->json([
                'text' => 'Sub Group has been created.'
            ]);
        } catch (\Throwable $th) {
            return $th;
        }
    }

    public function fetch(Request $req)
    {
        
        $role = Role::where('id', $req->parent_role_id)->with(['subRoles', 'roleHasSystem'])->first();
        if ($role) {
            $this->logCustomMessage('view_group', $role, Auth::user()->name . ' viewed group ' . $role->name, $role, 'View a group', new Activity());
        }
        return $role;
    }

    public function isRoot(Request $req)
    {
        return SubRole::where('sub_role_id', $req->sub_role_id)->first() ? false : true;
    }
}
