<?php

namespace App\Http\Controllers\AccessManagement\GroupManagement;

use Activity;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\AccessManagement\GroupManagement\Permission;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Validator;
use App\Models\AccessManagement\GroupManagement\RoleHasPermission;
use App\Models\AccessManagement\GroupManagement\RoleHasSystemModule;
use App\Models\AccessManagement\GroupManagement\SubRole;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Enums\Log\GroupLogType;

class RoleSystemModuleController extends Controller
{

    use HasCustomLogs;

    public function create(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'role_id' => 'required|exists:' . env('DB_CONNECTION') . '.roles,id',
            'role_system_module' => 'array',
        ]);
        DB::beginTransaction();
        try {
            if ($valid->fails()) {

                return response()->json([
                    'errors'    =>  $valid->errors()
                ], 400);
            }

            $this->assignModule($request->role_id, $request->role_system_module);

            DB::commit();
            Artisan::call('cache:clear');
            return response()->json([
                'text'  =>  'Role system module has been created.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning modules.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function assignModule($roleId, $roleSystemModules, $assignToSubRole = false)
    {

        $currentRoleSystemModule = RoleHasSystemModule::where('role_id', $roleId);

        $assignSystemModuleIds = collect($roleSystemModules)->pluck('system_module_id');

        $removedSystemModuleIds = $currentRoleSystemModule->whereNotIn('system_module_id', $assignSystemModuleIds)->get()->pluck('id');

        $allPermissions = [];

        foreach ($removedSystemModuleIds as $id) {
            $roleHasSystemModule = RoleHasSystemModule::where('id', $id)->with(['role' => function ($query) {
                $query->without([
                    'roleLeader',
                    'sectionAccess',
                    'member'
                ])->with(['roleHasSystem' => function ($query) {
                    $query->without([
                        'role',
                        'system'
                    ]);
                }]);
            }, 'systemModule' => function ($query) {
                $query->without([
                    'permission'
                ]);
            }])->first();

            $this->logCustomMessage('remove_group_module', $roleHasSystemModule, Auth::user()->name . ' removes a module ' . $roleHasSystemModule->systemModule->name . ' to the group ' . $roleHasSystemModule->role->name, $roleHasSystemModule, GroupLogType::REMOVE_GROUP_MODULE, new Activity());
        }

        $currentRoleSystemModule->delete();

        foreach ($roleSystemModules as $value) {
            $valid = Validator::make($value, [
                'system_module_id' => 'required|exists:' . env('DB_CONNECTION') . '.system_modules,id',
                'permissions' => 'array'
            ]);
            if ($valid->fails()) {
                return response()->json([
                    'errors'    =>  $valid->errors()
                ], 400);
            }

            $roleHasPermission = [];

            $where = [
                'role_id' => $roleId,
                'system_module_id' => $value['system_module_id']
            ];

            $roleSystemModule = RoleHasSystemModule::where($where)->with(['role' => function ($query) {
                $query->without([
                    'roleLeader',
                    'sectionAccess',
                    'member'
                ])->with(['roleHasSystem' => function ($query) {
                    $query->without([
                        'role',
                        'system'
                    ]);
                }]);
            }, 'systemModule' => function ($query) {
                $query->without([
                    'permission'
                ]);
            }])->without(['roleHasPermission'])->first();

            if (!$roleSystemModule) {
                RoleHasSystemModule::create($where);

                $roleSystemModule = RoleHasSystemModule::where($where)->with(['role' => function ($query) {
                    $query->without([
                        'roleLeader',
                        'sectionAccess',
                        'member'
                    ])->with(['roleHasSystem' => function ($query) {
                        $query->without([
                            'role',
                            'system'
                        ]);
                    }]);
                }, 'systemModule' => function ($query) {
                    $query->without([
                        'permission'
                    ]);
                }])->without(['roleHasPermission'])->first();

                $this->logCustomMessage('assign_group_module', $roleSystemModule, Auth::user()->name . ' assigns a module ' . $roleSystemModule->systemModule->name . ' to the group ' . $roleSystemModule->role->name, $roleSystemModule, GroupLogType::ASSIGN_GROUP_MODULE, new Activity());
            }

            $removedPermissions = RoleHasPermission::where([
                ['role_id', $roleId],
                ['role_system_module_id', $roleSystemModule->id]
            ])->whereNotIn('permission_id', $value['permissions'])->with(['permission'])->get();

            $currentPermissions = RoleHasPermission::where([
                ['role_id', $roleId],
                ['role_system_module_id', $roleSystemModule->id]
            ])->with(['permission'])->get()->pluck('permission_id')->toArray();

            foreach ($removedPermissions as $permission) {
                $roleSystemModule->permission = $permission['permission'];
                $this->logCustomMessage('remove_group_module_permission', $roleSystemModule, Auth::user()->name . ' removes a module permission ' . $permission['permission']['display_name'] . ' (module: ' . $roleSystemModule->systemModule->name . ') to the group ' . $roleSystemModule->role->name, $roleSystemModule, GroupLogType::REMOVE_GROUP_MODULE_PERMISSION, new Activity());
            }

            foreach ($value['permissions'] as $permission) {
                array_push($roleHasPermission, array(
                    'permission_id' => $permission,
                    'role_id' => $roleId,
                    'role_system_module_id' => $roleSystemModule->id
                ));
                $allPermissions[] = $permission;
                if (!in_array($permission, $currentPermissions)) {

                    $newPermission = Permission::find($permission);
                    $roleSystemModule->permission = $newPermission;
                    $this->logCustomMessage('assign_group_module_permission', $roleSystemModule, Auth::user()->name . ' assigns a module permission ' . $newPermission->display_name . ' (module: ' . $roleSystemModule->systemModule->name . ') to the group ' . $roleSystemModule->role->name, $roleSystemModule, GroupLogType::ASSIGN_GROUP_MODULE_PERMISSION, new Activity());   
                }
            }

            RoleHasPermission::where([
                ['role_id', $roleId],
                ['role_system_module_id', $roleSystemModule->id]
            ])->delete();

            RoleHasPermission::insert($roleHasPermission);
        }

        if ($assignToSubRole || count($roleSystemModules) <= 0) {
            $subRoles = SubRole::where('parent_role_id', $roleId)->get();
            foreach ($subRoles as $subRole) {
                $this->assignModule($subRole['sub_role_id'], $roleSystemModules);
            }
        } else {
            $this->syncSubRolePermission($roleId, $allPermissions, array_column($roleSystemModules, 'system_module_id'));
        }
    }

    public function syncSubRolePermission($roleId, $permissions, $systemModules)
    {
        $subRoles = SubRole::where('parent_role_id', $roleId)->get();
        foreach ($subRoles as $subRole) {
            RoleHasPermission::where('role_id', $subRole['sub_role_id'])->whereNotIn('permission_id', $permissions)->delete();
            RoleHasSystemModule::where('role_id', $subRole['sub_role_id'])->whereNotIn('system_module_id', $systemModules)->delete();
            $permissions = RoleHasPermission::where('role_id', $subRole['sub_role_id'])->get()->pluck('permission_id')->toArray();
            $systemModules = RoleHasSystemModule::where('role_id', $subRole['sub_role_id'])->get()->pluck('system_module_id')->toArray();
            $this->syncSubRolePermission($subRole['sub_role_id'], $permissions, $systemModules);
        }
    }

    public function delete(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('DB_CONNECTION') . '.role_has_system_modules,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                $valid->errors()
            ], 400);
        }
        try {

            RoleHasSystemModule::where('id', $request->id)->delete();
            Artisan::call('cache:clear');
            return response()->json([
                'message' => 'System is deleted'
            ], 200);
        } catch (\Throwable $th) {
            return $th;
        }
    }

    public function fetchByRoleID(Request $request)
    {
        return RoleHasSystemModule::where('role_id', $request->role_id)->get();
    }
}
