<?php

namespace App\Http\Controllers\AccessManagement\GroupManagement;

use Activity;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Models\AccessManagement\GroupManagement\RoleLeader;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Enums\Log\GroupLogType;

class RoleLeaderController extends Controller
{

    use HasCustomLogs;

    private $roleTable = 'roles';
    private $userTable = 'users';

    public function create(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'role_id' => 'required|exists:' . env('DB_CONNECTION') . '.' . $this->roleTable . ',id',
            'user_id' => 'required|exists:' . env('DB_CONNECTION') . '.' . $this->userTable . ',id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $roleLeader = RoleLeader::create([
                'role_id' => $request->role_id,
                'user_id' => $request->user_id
            ]);

            DB::commit();

            $this->logCustomMessage(
                'assign_group_leader',
                $roleLeader,
                Auth::user()->name . ' assigns ' . $roleLeader->user->name . ' as a group leader to the ' . $roleLeader->role->name,
                $roleLeader,
                GroupLogType::ASSIGN_GROUP_LEADER,
                new Activity()
            );
            return response()->json([
                'text'  =>  'Role leader has been created.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating role leader.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required',
            'new_role_id' => 'required|exists:' . env('DB_CONNECTION') . '.' . $this->roleTable . ',id',
            'new_user_id' => 'required|exists:' . env('DB_CONNECTION') . '.' . $this->userTable . ',id',
            'old_role_id' => 'required|exists:' . env('DB_CONNECTION') . '.' . $this->roleTable . ',id',
            'old_user_id' => 'required|exists:' . env('DB_CONNECTION') . '.' . $this->userTable . ',id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $roleLeader = RoleLeader::where([
                ['id', $request->id],
                ['role_id', $request->old_role_id],
                ['user_id', $request->old_user_id]
            ]);

            $oldRoleLeader = clone $roleLeader->first();

            $roleLeader->update([
                'role_id' => $request->new_role_id,
                'user_id' => $request->new_user_id
            ]);

            $roleLeader = RoleLeader::where([
                'role_id' => $request->new_role_id,
                'user_id' => $request->new_user_id
            ])->first();

            $roleLeader->attributes = collect($roleLeader);
            $roleLeader->old = collect($oldRoleLeader);

            DB::commit();

            if ($oldRoleLeader->user->id != $roleLeader->user->id) {
                $this->logCustomMessage(
                    'update_group_leader',
                    $roleLeader,
                    Auth::user()->name . ' removes ' . $oldRoleLeader->user->name . ' and assigns ' . $roleLeader->user->name . ' as a new leader to the ' . $roleLeader->role->name,
                    $roleLeader,
                    GroupLogType::UPDATE_GROUP_LEADER,
                    new Activity()
                );
            }

            return response()->json([
                'text'  =>  'Role leader has been updated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating role leader.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }
}
