<?php

namespace App\Http\Controllers\AccessManagement\GroupManagement;

use Activity;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Models\AccessManagement\GroupManagement\Role;
use App\Models\AccessManagement\GroupManagement\System;
use App\Models\AccessManagement\GroupManagement\ModelHasRole;
use App\Models\AccessManagement\GroupManagement\RoleHasSystem;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Enums\Log\GroupLogType;
use Suiterus\Dms\Models\Repositories\SectionAccess;

class RoleHasSystemController extends Controller
{

    use HasCustomLogs;

    public function create(Request $request)
    {
        $isSystemIdValid = Validator::make($request->all(), [
            'system_id' => 'required|exists:' . env('DB_CONNECTION') . '.systems,id'
        ]);

        if ($isSystemIdValid->fails()) {
            return response()->json([
                'errors'    =>  $isSystemIdValid->errors()
            ], 400);
        }

        $system = System::where('id', $request->system_id)->first();
        $oldGroupName = $request->name;
        $request->name = $request->name . ' ' . $system->name;

        $isName = Validator::make(['name' => $request->name], [
            'name'  =>  'required|string|unique:roles',
        ]);

        if ($isName->fails()) {
            return response()->json([
                'errors'    =>  $isName->errors()
            ], 400);
        }


        DB::beginTransaction();
        try {
            $role = Role::where('name', $request->name)->first();
            $system = System::where('id', $request->system_id)->first();

            if (!$role) {
                $role = Role::create([
                    'name'  =>  $request->name,
                    'guard_name' =>  'api'
                ]);

                $role->system = $system;

                $this->logCustomMessage('create_group', $role, Auth::user()->name . ' has created a group called ' . $oldGroupName, $role, GroupLogType::CREATE, new Activity());
            }

            RoleHasSystem::create([
                'system_id' => $request->system_id,
                'role_id' => $role->id
            ]);

            foreach ($request->group_member as $member) {

                ModelHasRole::create([
                    'role_id' => $role->id,
                    'model_type' => 'App\Models\User',
                    'model_id' => $member,
                ]);


                $newMember = User::without([
                    'roles',
                    'permissions',
                    'storage',
                    'employeeMetaInfo',
                    'supervisor',
                    'exitInterview',
                    'userProfilePicture',
                    'profileBasicInfo'
                ])->where('id', $member)->first();

                $role = Role::where('id', $role->id)->without([
                    'roleLeader',
                    'sectionAccess',
                    'member'
                ])->with(['roleHasSystem' => function ($query) {
                    $query->without(['role']);
                }])->first();

                $newMember->role = $role;

                $this->logCustomMessage(
                    'add_group_member',
                    $newMember,
                    Auth::user()->name . ' added ' . $newMember->name . ' to the group ' . $oldGroupName,
                    $newMember,
                    GroupLogType::ADD_GROUP_MEMBER,
                    new Activity()
                );
            }
            DB::commit();
            return response()->json([
                'text'  =>  'Role has been created.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating new Role.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required',
            'name'  =>  'required|string|min:4'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $system = System::where('id', $request->system_id)->first();

            Role::where('id', $request->id)->update([
                'name' => $request->name . ' ' . $system->name
            ]);


            DB::commit();
            return response()->json([
                'text'  =>  'Role has been updated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating new Role.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete(Request $request)
    {
    }

    public function fetchBySystem(Request $request)
    {
        return RoleHasSystem::when($request->system_id, function ($query) use ($request) {
            $query->where('system_id', $request->system_id);
        })->get();
    }

    public function paginate(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        $roleHasSystemPaginate = RoleHasSystem::when($request->system_id, function ($query) use ($request) {
            $query->where('system_id', $request->system_id);
        })->with(['role' => function ($query) {
            $query->with(['subRoles']);
        }])->paginate($paginate);

        $roleHasSystemPaginate = json_decode(json_encode($roleHasSystemPaginate));

        for ($i = 0; $i < count($roleHasSystemPaginate->data); $i++) {
            $roleID = $roleHasSystemPaginate->data[$i]->role->id;
            $value = ModelHasRole::where('role_id', $roleID)->take(4)->get();
            $roleHasSystemPaginate->data[$i]->role->model_has_role = $value;
            $isMemberMorethanFour = count(ModelHasRole::where('role_id', $roleID)->take(5)->get()) > 4;
            $value = SectionAccess::where('group_id', $roleID)->take(3)->get();
            $roleHasSystemPaginate->data[$i]->role->section_access = $value;
            $isSectionMoreThanThree = count(SectionAccess::where('group_id', $roleID)->take(4)->get()) > 3;
            $roleHasSystemPaginate->data[$i]->role->is_member_more_than_four = $isMemberMorethanFour;
            $roleHasSystemPaginate->data[$i]->role->is_section_more_than_three = $isSectionMoreThanThree;
        }

        return $roleHasSystemPaginate;
    }
}
