<?php

namespace App\Http\Controllers\AccessManagement\GroupManagement;

use Activity;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Models\AccessManagement\GroupManagement\Role;
use App\Models\AccessManagement\GroupManagement\System;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Enums\Log\GroupLogType;

class RoleController extends Controller
{

    use HasCustomLogs;

    public function create(Request $request){
        $valid = Validator::make($request->all(),[
            'name'  =>  'required|string|unique:roles'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{

            Role::create([
                'name'  =>  $request->name,
                'guard_name' =>  'api'
            ]);

            DB::commit();
            return response()->json([
                'text'  =>  'Role has been created.'
            ]);
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in creating new Role.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function show($id){
        return Role::where('id', $id)->with(['roleHasPermission'])->first();
    }

    public function delete(Request $request){
        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:table,column'
        ]);

        if ($valid->fails()) {
            return response()->json([
                $valid->errors()
            ], 400);
        }
        try {
            Role::where('id', $request->id)->delete();
            return response()->json([
                'message' => 'Group is deleted'
            ], 200);
        } catch (\Throwable $th) {
            return $th;
        }
    }

    public function update(Request $request){
        $valid = Validator::make($request->all(),[
            'id' => 'required',
            'name'  =>  'required|string|min:4|unique:roles'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{

            $system = System::where('id', $request->system_id)->first();

            $role = Role::where('id', $request->id);

            $oldRole = clone $role->first();

            $role->update([
                'name'  =>  $request->name . ' ' . $system->name,
            ]);

            $newRole = $role->first();

            $newRole->system = $system;
            $oldRole->system = $system;

            $newRole->attributes = collect($newRole);
            $newRole->old = collect($oldRole);

            DB::commit();

            $this->logCustomMessage('rename_group', $newRole, Auth::user()->name . ' has renamed the group '.$oldRole->name.' to ' . $newRole->name, $newRole, GroupLogType::RENAME, new Activity());

            return response()->json([
                'text'  =>  'Role has been updated.'
            ]);
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in creating new Role.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function fetch(){
        return Role::get();
    }

    public function paginate(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return Role::paginate($paginate);
    }
}
